#include "GlowShader.h"

GlowShader::GlowShader(ID3D11Device* device, HWND hwnd) : BaseShader(device, hwnd)
{
	initShader(L"postProcessing_vs.cso", L"glow_ps.cso");
}

GlowShader::~GlowShader()
{
	//Release the buffers
	if (glowBuffer)
	{
		glowBuffer->Release();
		glowBuffer = 0;
	}

	if (matrixBuffer)
	{
		matrixBuffer->Release();
		matrixBuffer = 0;
	}

	//Release the sampler
	if (sampleState)
	{
		sampleState->Release();
		sampleState = 0;
	}

	if (layout)
	{
		layout->Release();
		layout = 0;
	}

	BaseShader::~BaseShader();
}

void GlowShader::initShader(const wchar_t* vsFilename, const wchar_t* psFilename)
{
	loadVertexShader(vsFilename);
	loadPixelShader(psFilename);

	//Setup the matrix buffer
	D3D11_BUFFER_DESC matrixBufferDesc;
	matrixBufferDesc.Usage = D3D11_USAGE_DYNAMIC;
	matrixBufferDesc.ByteWidth = sizeof(MatrixBufferType);
	matrixBufferDesc.BindFlags = D3D11_BIND_CONSTANT_BUFFER;
	matrixBufferDesc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;
	matrixBufferDesc.MiscFlags = 0;
	matrixBufferDesc.StructureByteStride = 0;
	renderer->CreateBuffer(&matrixBufferDesc, NULL, &matrixBuffer);

	//Setup the glow buffer
	D3D11_BUFFER_DESC glowBufferDesc;
	glowBufferDesc.Usage = D3D11_USAGE_DYNAMIC;
	glowBufferDesc.ByteWidth = sizeof(GlowIntensityBufferType);
	glowBufferDesc.BindFlags = D3D11_BIND_CONSTANT_BUFFER;
	glowBufferDesc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;
	glowBufferDesc.MiscFlags = 0;
	glowBufferDesc.StructureByteStride = 0;
	renderer->CreateBuffer(&glowBufferDesc, NULL, &glowBuffer);

	//Setup the base sampler
	D3D11_SAMPLER_DESC samplerDesc;
	samplerDesc.Filter = D3D11_FILTER_ANISOTROPIC;
	samplerDesc.AddressU = D3D11_TEXTURE_ADDRESS_WRAP;
	samplerDesc.AddressV = D3D11_TEXTURE_ADDRESS_WRAP;
	samplerDesc.AddressW = D3D11_TEXTURE_ADDRESS_WRAP;
	samplerDesc.MipLODBias = 0.0f;
	samplerDesc.MaxAnisotropy = 1;
	samplerDesc.ComparisonFunc = D3D11_COMPARISON_ALWAYS;
	samplerDesc.MinLOD = 0;
	samplerDesc.MaxLOD = D3D11_FLOAT32_MAX;
	renderer->CreateSamplerState(&samplerDesc, &sampleState);
}

void GlowShader::setShaderParameters(ID3D11DeviceContext* deviceContext, const XMMATRIX &worldMatrix, const XMMATRIX &viewMatrix, const XMMATRIX &projectionMatrix, ID3D11ShaderResourceView* sceneTexture, XMFLOAT4 intensity)
{
	XMMATRIX tworld, tview, tproj;
	D3D11_MAPPED_SUBRESOURCE mappedResource;

	tworld = XMMatrixTranspose(worldMatrix);
	tview = XMMatrixTranspose(viewMatrix);
	tproj = XMMatrixTranspose(projectionMatrix);

	//Map the matrix buffer
	deviceContext->Map(matrixBuffer, 0, D3D11_MAP_WRITE_DISCARD, 0, &mappedResource);
	MatrixBufferType* matrixPtr = (MatrixBufferType*)mappedResource.pData;
	matrixPtr->world = tworld;
	matrixPtr->view = tview;
	matrixPtr->projection = tproj;
	deviceContext->Unmap(matrixBuffer, 0);
	deviceContext->VSSetConstantBuffers(0, 1, &matrixBuffer);

	//Map the glow buffer
	deviceContext->Map(glowBuffer, 0, D3D11_MAP_WRITE_DISCARD, 0, &mappedResource);
	GlowIntensityBufferType* glowPtr = (GlowIntensityBufferType*)mappedResource.pData;
	glowPtr->intensity = intensity;
	deviceContext->Unmap(glowBuffer, 0);

	//Set the pixel shader textures, buffers and samplers
	deviceContext->PSSetConstantBuffers(0, 1, &glowBuffer);
	deviceContext->PSSetShaderResources(0, 1, &sceneTexture);
	deviceContext->PSSetSamplers(0, 1, &sampleState);
}
